﻿
CREATE PROCEDURE [PanelMgmt].[usp_RebuildAllCDWPatients] 
AS
--==================================================================================
--Requestor             : Non Va Coordinated Care Project, but produces a great
--                        general use all patient table for any application.
--Author                : Jerry Kohler, Jianji Yang - Northwest Innovation Center
--Object/SP Name        : usp_RebuildAllCDWPatients
--Server                : SERVER       
--Data Base             : CC_REFDOC
--Schema                : PanelMgmt
--Report                : N/A
--Folder                : N/A
--Report Loc            : N/A
--Job                   : None
--Job Loc               : None
--Note                  : This table contains ALL patients, including
--                        deceased and test.
--Date Created          : 03-10-2016
--
--
--Purpose               : Because SPatient does not have an index on PatientSSN,
--                        we create a table to translate between several types of
--                        patient identification for REFDOC:
--                           - SSN
--                           - SID
--                           - ICN
--                           - Name (last,first)
--                           - CPRSKey (first character of last name, last4)
--
--Last Changed          : 08-10-2017
--Last Changed By		: Brian Diggs - NWIC
--Reason For Change		: Process was still blocking for way too long, preventing
--                        lookups of patients. Refactored based on discussions
--                        with Andrew Kelly. Extract the batches outside the
--                        (blocking) MERGE, then filter them while
--                        flagging if the record is an update or insert.
--                        Do the updates and then inserts and update the ETLBatchID
--
--Last Changed          : 07-14-2017
--Last Changed By		: Brian Diggs - NWIC
--Reason For Change		: Implemented suggestions from Andrew Kelly
--
--Last Changed          : 06-09-2017
--Last Changed By		: Brian Diggs - NWIC
--Reason For Change		: Converted to incremental update following recommended CDW 
--                        paradigm
--
--Last Changed          : 04-19-2017
--Last Changed By		: Brian Diggs - NWIC
--Reason For Change		: Stripped this down to just the parts needed for REFDOC
--                        as part of the migration to the Field Reporting Enclave.
--                        The only columns needed are Sta3n, PatientSSN, PatientICN,
--                        PatientSID, and PatientIEN.
--
--Last Changed          : 06-06-2016
--Last Changed By		: Jerry Kohler - NWIC
--Reason For Change		: Added Next of Kin, and per Microsoft recommendation
--                        changed drop/create table to drop indexes and TRUNCATE.
--                        Also added WITH (TABLOCK) in the insert.
--                        All of this should boost the performance, and not
--                        eliminate stored execution plans on the table.
--
--Last Changed          : 06-03-2016
--Last Changed By		: Jerry Kohler - NWIC
--Reason For Change		: Take all patients from SPatient, test or deceased.
--                        Join to PanelMgmt.PMMasterCohortDeceasedIndexNEW
--                        to ensure better accuracy.  Add deceased flag and
--                        date of death.

--Last Changed          : 04-05-2016
--Last Changed By		: Jianji Yang - NWIC
--Reason For Change		: Add test patient data, deceased patients, and patient's
--                      : home station (based on PCP's Sta3n).
--
--Last Changed          : 03-17-2017
--Last Changed By		: Jerry Kohler - NWIC
--Reason For Change		: Add PatientIEN so that function App.udf_AllPatientsSIDs
--                      : can now hit AllCDWPatients instead of SPatient.  This
--                      : table is much smaller, and will be much faster.
--
--Last Changed          : 03-20-2017
--Last Changed By		: Jerry Kohler - NWIC
--Reason For Change		: Change procedure to place data in a temporary table THEN
--                      : truncate and insert into the production table.  This
--                      : minimize the amount of time queries would return no data.
--==================================================================================

BEGIN

	SET NOCOUNT ON;

	--declare batch variables and cursor
	DECLARE @ETLBatchID AS INT, @ExtractBatchID AS INT;
	DECLARE @DWViewName AS VARCHAR(100) = 'SPatient';

	--add a TRY/CATCH to exit upon failure
	BEGIN TRY

		SET @ExtractBatchID = (SELECT EL.LastExtractBatchID FROM Dflt.ExtractBatchLog AS EL WHERE EL.DWViewName = @DWViewName) ;

		DECLARE BatchCursor CURSOR STATIC 
		FOR
		--get batch ids
		SELECT EX.ETLBatchID, EX.ExtractBatchID
		FROM SPV.EB.ExtractBatch AS EX
		WHERE EX.DWViewName = @DWViewName
			AND EX.ExtractBatchID > @ExtractBatchID
		ORDER BY EX.ExtractBatchID;

		CREATE TABLE #spat1 (
			Sta3n smallint,
			PatientIEN varchar(50),
			PatientSSN varchar(50),
			PatientSID int,
			PatientICN varchar(50)
		)
		DECLARE @LastExtractBatchID AS INT;

		OPEN BatchCursor;
		FETCH NEXT FROM BatchCursor INTO @ETLBatchID, @ExtractBatchID;

		WHILE @@FETCH_STATUS = 0
		BEGIN
			INSERT INTO #spat1
			SELECT Sta3n, PatientIEN, PatientSSN, PatientSID, PatientICN
			FROM SPV.SPatient.SPatient
			WHERE ETLBatchID = @ETLBatchID

			-- Store to update tracking table later
			SET @LastExtractBatchID = @ExtractBatchID

			--merge data from view into production(destination) table
			--grab the next batch for processing
			FETCH NEXT FROM BatchCursor INTO @ETLBatchID, @ExtractBatchID;
		END

		--close and deallocate the cursor once all batches for a given table have been processed
		CLOSE BatchCursor;
		DEALLOCATE BatchCursor;

		SELECT
			S.Sta3n,
			S.PatientIEN,
			S.PatientSSN,
			S.PatientSID,
			S.PatientICN,
			CASE
				WHEN A.PatientSID IS NULL THEN 'I'
				ELSE 'U'
			END AS InsertUpdate
		INTO #spat2
		FROM
			#spat1 AS S
			LEFT JOIN App.AllCDWPatients AS A WITH (NOLOCK)
				ON S.PatientSID = A.PatientSID
			WHERE
				A.PatientSID IS NULL 
				OR (S.Sta3n != A.Sta3n
					OR S.PatientICN != A.PatientICN
					OR S.PatientSSN != A.PatientSSN
					OR S.PatientIEN != A.PatientIEN)

		CREATE CLUSTERED INDEX [#tmpidx] ON #spat2 (PatientSID ASC) WITH (DATA_COMPRESSION = PAGE);

		BEGIN TRANSACTION
			UPDATE A
			SET
				A.Sta3n = U.Sta3n,
				A.PatientIEN = U.PatientIEN,
				A.PatientSSN = U.PatientSSN,
				A.PatientICN = U.PatientICN
			FROM
				App.AllCDWPatients AS A
				INNER JOIN #spat2 AS U
					ON A.PatientSID = U.PatientSID
			WHERE
				U.InsertUpdate = 'U'

			INSERT INTO App.AllCDWPatients (Sta3n, PatientIEN, PatientSSN, PatientSID, PatientICN)
			SELECT Sta3n, PatientIEN, PatientSSN, PatientSID, PatientICN
			FROM #spat2
			WHERE InsertUpdate = 'I'

			--after merging batch then update the ExtractBatchLog table
			UPDATE Dflt.ExtractBatchLog
				SET LastExtractBatchID = @LastExtractBatchID, LastExtractDateTime = GETDATE()
			WHERE DWViewName = @DWViewName;

		COMMIT TRANSACTION

	END TRY

	--CATCH will report back specific details about failure; the loop will stop
	BEGIN CATCH

		IF @@TRANCOUNT > 0
			ROLLBACK TRANSACTION;

		SELECT
		ERROR_NUMBER() AS ErrorNumber,
		ERROR_SEVERITY() AS ErrorSeverity,
		ERROR_STATE() AS ErrorState,
		ERROR_PROCEDURE() AS ErrorProcedure,
		ERROR_LINE() AS ErrorLine,
		ERROR_MESSAGE() AS ErrorMessage;
         
	END CATCH 

END
